import http.server
import socketserver
import os
import json
import socket
import cgi
import sys
import shutil
import urllib.parse
from datetime import datetime
import mimetypes
import traceback
import webbrowser
import threading

# --- パス設定 ---
SYSTEM_DIR = os.path.dirname(os.path.abspath(__file__))
ROOT_DIR = os.path.dirname(SYSTEM_DIR)

SHARED_DIR_NAME = '配布資料'
SUBMIT_DIR_NAME = '提出ボックス'

SHARED_DIR = os.path.join(ROOT_DIR, SHARED_DIR_NAME)
SUBMIT_DIR = os.path.join(ROOT_DIR, SUBMIT_DIR_NAME)
CONFIG_FILE = os.path.join(SYSTEM_DIR, 'config.json')
LOG_FILE = os.path.join(SYSTEM_DIR, 'server.log')
ICON_FILE = os.path.join(SYSTEM_DIR, 'classpost_icon.png')

os.makedirs(SHARED_DIR, exist_ok=True)
os.makedirs(SUBMIT_DIR, exist_ok=True)

DEFAULT_PORT = 8029
CURRENT_PORT = DEFAULT_PORT
ADMIN_PASSWORD = 'admin'  # 管理者パスワード

# --- stderr(エラー出力)をログファイルに退避 ---
# Windows コマンドプロンプトで日本語がstderrに出るとクラッシュするため
try:
    sys.stderr = open(LOG_FILE, 'a', encoding='utf-8')
except Exception:
    pass


def safe_print(msg):
    """コンソールに安全に表示する。日本語はすべて?に置換してクラッシュ防止。"""
    try:
        safe_msg = str(msg).encode('ascii', errors='replace').decode('ascii')
        sys.stdout.write(safe_msg + '\n')
        sys.stdout.flush()
    except Exception:
        pass

# --- HTML テンプレート (Class Postデザイン) ---
# ※ オフライン環境で動作するため、外部フォント(Google Fonts)は使わない
HTML_HEADER = """
<!DOCTYPE html>
<html lang="ja">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Class Post</title>
    <link rel="icon" type="image/png" href="/favicon.png">
    <style>
        :root {
            --primary-color: #2c3e50;
            --accent-color: #3498db;
            --success-color: #27ae60;
            --danger-color: #e74c3c;
            --bg-color: #f0f2f5;
            --card-bg: #ffffff;
            --text-color: #333333;
            --border-radius: 12px;
        }
        body {
            font-family: 'Meiryo', 'Yu Gothic UI', 'Hiragino Kaku Gothic ProN', 'Segoe UI', sans-serif;
            background-color: var(--bg-color);
            margin: 0;
            padding: 20px;
            color: var(--text-color);
            line-height: 1.6;
        }
        .container {
            max-width: 800px;
            margin: 0 auto;
            background: var(--card-bg);
            padding: 40px;
            border-radius: var(--border-radius);
            box-shadow: 0 4px 15px rgba(0,0,0,0.05);
        }
        h1 {
            color: var(--primary-color);
            font-size: 28px;
            border-bottom: 3px solid var(--accent-color);
            padding-bottom: 15px;
            margin-bottom: 30px;
            text-align: center;
            letter-spacing: 1px;
        }
        h2, h3 { color: var(--primary-color); margin-top: 30px; margin-bottom: 15px; }
        
        .btn {
            display: inline-block; padding: 12px 25px; font-size: 16px; font-weight: bold; text-align: center; text-decoration: none; border-radius: 50px; cursor: pointer; border: none; transition: transform 0.1s, box-shadow 0.2s; box-shadow: 0 2px 5px rgba(0,0,0,0.1);
        }
        .btn:hover { transform: translateY(-2px); box-shadow: 0 4px 8px rgba(0,0,0,0.15); }
        .btn-primary { background-color: var(--accent-color); color: white; }
        .btn-success { background-color: var(--success-color); color: white; }
        .btn-danger { background-color: var(--danger-color); color: white; }
        .btn-secondary { background-color: #95a5a6; color: white; }
        .btn-outline { background-color: transparent; border: 2px solid #95a5a6; color: #7f8c8d; }
        .btn-outline:hover { background-color: #95a5a6; color: white; }
        
        input[type="text"], input[type="password"], input[type="file"] {
            width: 100%; padding: 12px; font-size: 16px; border: 2px solid #ddd; border-radius: 8px; box-sizing: border-box; margin-bottom: 15px;
        }
        input:focus { border-color: var(--accent-color); outline: none; }

        .card-list { list-style: none; padding: 0; }
        .file-card {
            background: #f8f9fa; border: 1px solid #eee; border-radius: 10px; padding: 15px 20px;
            margin-bottom: 15px; display: flex; justify-content: space-between; align-items: center;
        }
        .file-card:hover { background: #f1f4f6; }
        .file-info { display: flex; align-items: center; gap: 15px; font-size: 16px; font-weight: 500; }
        .file-icon { font-size: 24px; }
        
        .status-tag { font-size: 12px; padding: 4px 8px; border-radius: 4px; font-weight: bold; margin-left:10px; }
        .tag-open { background: #d4edda; color: #155724; }
        .tag-closed { background: #f8d7da; color: #721c24; }
        
        .disabled-card { opacity: 0.6; background: #eee; pointer-events: none; }
        .upload-area { background: #eef6fc; border: 2px dashed #bbd6ea; padding: 20px; border-radius: 10px; margin-bottom: 30px; text-align: center; }
        
        .nav-top { margin-bottom: 20px; display:flex; justify-content: space-between; align-items: center; }
        .alert { padding: 15px; margin-bottom: 20px; border-radius: 8px; font-weight: bold; text-align: center;}
        .alert-success { background: #d4edda; color: #155724; }
        .alert-error { background: #f8d7da; color: #721c24; }
    </style>
</head>
<body>
<div class="container">
"""

HTML_FOOTER = """
</div>
<div style="position:fixed; bottom:4px; right:8px; font-size:9px; color:#f0f0f0;">seyaseya.org</div>
</body>
</html>
"""

# --- サーバーロジック ---

def load_config():
    default = {"locked_files": []}
    if not os.path.exists(CONFIG_FILE):
        return default
    try:
        with open(CONFIG_FILE, 'r', encoding='utf-8') as f:
            config = json.load(f)
        # locked_files キーが無い場合は補完
        if 'locked_files' not in config:
            config['locked_files'] = []
        return config
    except Exception:
        return default

def save_config(config):
    with open(CONFIG_FILE, 'w', encoding='utf-8') as f:
        json.dump(config, f, indent=4)

def get_ip_address():
    s = socket.socket(socket.AF_INET, socket.SOCK_DGRAM)
    try:
        s.connect(('10.255.255.255', 1))
        IP = s.getsockname()[0]
    except Exception:
        IP = '127.0.0.1'
    finally:
        s.close()
    return IP

class ClassroomHandler(http.server.BaseHTTPRequestHandler):

    def log_message(self, format, *args):
        """アクセスログを stderr(=ログファイル) に書く。コンソールには出さない。"""
        try:
            sys.stderr.write("%s - [%s] %s\n" % (
                self.client_address[0],
                self.log_date_time_string(),
                format % args))
            sys.stderr.flush()
        except Exception:
            pass
    
    def do_GET(self):
        try:
            self._handle_GET()
        except Exception:
            try:
                traceback.print_exc()
            except Exception:
                pass

    def _handle_GET(self):
        parsed_path = urllib.parse.urlparse(self.path)
        path = parsed_path.path
        query = urllib.parse.parse_qs(parsed_path.query)

        # 生徒用画面 (パスワード不要)
        if path == '/':
            self.render_student_page()

        # 教員用ログイン画面
        elif path == '/admin':
            self.send_response(200)
            self.send_header('Content-type', 'text/html; charset=utf-8')
            self.end_headers()
            html = HTML_HEADER + """
            <h1>Class Post - 教員用ログイン</h1>
            <div style="text-align:center; margin-bottom:20px; color:#666;">
                <p>管理機能を利用するにはパスワードが必要です。</p>
            </div>
            <form action="/manager" method="GET" style="max-width:400px; margin:0 auto;">
                <label style="font-weight:bold; display:block; margin-bottom:5px;">パスワード</label>
                <input type="password" name="pwd" required placeholder="パスワードを入力">
                <button type="submit" class="btn btn-primary" style="width:100%;">ログイン</button>
            </form>
            <div style="text-align:center; margin-top:30px;">
                <a href="/" class="btn btn-outline" style="font-size:14px;">生徒用画面に戻る</a>
            </div>
            """ + HTML_FOOTER
            self.wfile.write(html.encode('utf-8'))

        # 教員用管理画面 (パスワード必須)
        elif path == '/manager':
            pwd = query.get('pwd', [''])[0]
            if pwd != ADMIN_PASSWORD:
                self.send_error(403, "Forbidden: Password Incorrect")
                return
            self.render_teacher_page()

        # ファイルダウンロード
        elif path.startswith('/download/'):
            filename = urllib.parse.unquote(path[10:])
            file_path = os.path.join(SHARED_DIR, filename)
            config = load_config()

            # ロック中のファイルはダウンロード禁止
            if filename in config['locked_files']:
                self.send_error(403, "This file is not available yet")
                return

            if os.path.exists(file_path):
                self.send_response(200)
                mime_type, _ = mimetypes.guess_type(file_path)
                if mime_type is None: mime_type = 'application/octet-stream'
                self.send_header('Content-type', mime_type)
                self.send_header('Content-Disposition', f'attachment; filename="{urllib.parse.quote(filename)}"')
                self.send_header('Content-Length', os.path.getsize(file_path))
                self.end_headers()
                with open(file_path, 'rb') as f:
                    shutil.copyfileobj(f, self.wfile)
            else:
                self.send_error(404, "File not found")

        # 提出物ダウンロード（教員用・パスワード必須）
        elif path.startswith('/submitted/'):
            pwd = query.get('pwd', [''])[0]
            if pwd != ADMIN_PASSWORD:
                self.send_error(403, "Forbidden")
                return
            filename = urllib.parse.unquote(path[11:])
            file_path = os.path.join(SUBMIT_DIR, filename)
            # パストラバーサル防止
            if os.path.commonpath([SUBMIT_DIR, os.path.abspath(file_path)]) != SUBMIT_DIR:
                self.send_error(403, "Forbidden")
                return
            if os.path.exists(file_path) and os.path.isfile(file_path):
                self.send_response(200)
                mime_type, _ = mimetypes.guess_type(file_path)
                if mime_type is None: mime_type = 'application/octet-stream'
                self.send_header('Content-type', mime_type)
                self.send_header('Content-Disposition', f'attachment; filename="{urllib.parse.quote(filename)}"')
                self.send_header('Content-Length', os.path.getsize(file_path))
                self.end_headers()
                with open(file_path, 'rb') as f:
                    shutil.copyfileobj(f, self.wfile)
            else:
                self.send_error(404, "File not found")

        # favicon
        elif path == '/favicon.png':
            if os.path.exists(ICON_FILE):
                self.send_response(200)
                self.send_header('Content-type', 'image/png')
                self.send_header('Content-Length', os.path.getsize(ICON_FILE))
                self.send_header('Cache-Control', 'public, max-age=86400')
                self.end_headers()
                with open(ICON_FILE, 'rb') as f:
                    shutil.copyfileobj(f, self.wfile)
            else:
                self.send_error(404, "Not Found")

        else:
            self.send_error(404, "Not Found")

    def do_POST(self):
        try:
            self._handle_POST()
        except Exception:
            try:
                traceback.print_exc()
            except Exception:
                pass

    def _handle_POST(self):
        parsed_path = urllib.parse.urlparse(self.path)
        path = parsed_path.path

        content_type = self.headers.get('Content-Type', '')

        # multipart/form-data の場合は cgi.FieldStorage を使う
        if content_type.startswith('multipart/form-data'):
            form = cgi.FieldStorage(
                fp=self.rfile,
                headers=self.headers,
                environ={'REQUEST_METHOD': 'POST'}
            )
        elif content_type.startswith('application/x-www-form-urlencoded'):
            # 通常のフォーム送信 (shutdown, toggle_permission など)
            length = int(self.headers.get('Content-Length', 0))
            body = self.rfile.read(length).decode('utf-8')
            form = urllib.parse.parse_qs(body)
        else:
            self.send_error(400, "Bad Request")
            return

        # 生徒の提出 (パスワード不要、multipart必須)
        if path == '/submit_file':
            if not isinstance(form, cgi.FieldStorage):
                self.send_error(400, "Bad Request")
                return
            student_name = form.getvalue("student_name")
            fileitem = form['file']
            if fileitem.filename and student_name:
                safe_filename = os.path.basename(fileitem.filename)
                save_name = f"{student_name}_{safe_filename}"
                save_path = os.path.join(SUBMIT_DIR, save_name)
                with open(save_path, 'wb') as f:
                    f.write(fileitem.file.read())
                self.redirect('/?msg=' + urllib.parse.quote('提出が完了しました！'))
            else:
                self.redirect('/?err=' + urllib.parse.quote('名前とファイルは必須です'))

        # --- 以下、教員用機能 (パスワード必須) ---
        
        elif path == '/upload_material':
            if not isinstance(form, cgi.FieldStorage):
                self.send_error(400, "Bad Request")
                return
            pwd = form.getvalue("pwd")
            if pwd != ADMIN_PASSWORD:
                self.send_error(403, "Forbidden")
                return

            fileitem = form['file']
            if fileitem.filename:
                fn = os.path.basename(fileitem.filename)
                save_path = os.path.join(SHARED_DIR, fn)
                with open(save_path, 'wb') as f:
                    f.write(fileitem.file.read())
            
            self.redirect(f'/manager?pwd={ADMIN_PASSWORD}')

        elif path == '/toggle_permission':
            pwd = self._get_form_value(form, "pwd")
            if pwd != ADMIN_PASSWORD:
                self.send_error(403, "Forbidden")
                return

            filename = self._get_form_value(form, "filename")
            config = load_config()
            if filename in config['locked_files']:
                config['locked_files'].remove(filename)
            else:
                config['locked_files'].append(filename)
            save_config(config)
            self.redirect(f'/manager?pwd={ADMIN_PASSWORD}')
        
        elif path == '/shutdown':
            pwd = self._get_form_value(form, "pwd")
            if pwd != ADMIN_PASSWORD:
                self.send_error(403, "Forbidden")
                return

            self.send_response(200)
            self.send_header('Content-type', 'text/html; charset=utf-8')
            self.end_headers()
            self.wfile.write("サーバーを停止しています...".encode('utf-8'))
            import threading
            def kill_me():
                os._exit(0)
            threading.Timer(1.0, kill_me).start()

    @staticmethod
    def _get_form_value(form, key):
        """cgi.FieldStorage と parse_qs の両方からフォーム値を取得する"""
        if isinstance(form, dict):
            # parse_qs は値をリストで返す
            vals = form.get(key, [''])
            return vals[0] if vals else ''
        else:
            return form.getvalue(key, '')

    def redirect(self, location):
        self.send_response(303)
        # Location ヘッダーは latin-1 のみ対応のため、非ASCII文字をURLエンコード
        safe_location = urllib.parse.quote(location, safe='/:?=&%+')
        self.send_header('Location', safe_location)
        self.end_headers()

    def render_student_page(self):
        config = load_config()
        if os.path.exists(SHARED_DIR):
            files = sorted(os.listdir(SHARED_DIR))
        else:
            files = []
        
        file_list_html = ""
        for f in files:
            if f.startswith('.'): continue
            is_locked = f in config['locked_files']
            if is_locked:
                file_list_html += f'''
                <li class="file-card disabled-card">
                    <div class="file-info"><span class="file-icon">🔒</span> {f} <span class="status-tag tag-closed">準備中</span></div>
                </li>'''
            else:
                file_list_html += f'''
                <li class="file-card">
                    <div class="file-info"><span class="file-icon">📄</span> {f} <span class="status-tag tag-open">公開中</span></div>
                    <a href="/download/{urllib.parse.quote(f)}" class="btn btn-success" style="font-size:13px; padding:6px 12px; min-width:50px;">DL</a>
                </li>'''

        msg = ""
        query = urllib.parse.urlparse(self.path).query
        q_dict = urllib.parse.parse_qs(query)
        if 'msg' in q_dict:
            msg = f'<div class="alert alert-success">{q_dict["msg"][0]}</div>'
        if 'err' in q_dict:
            msg = f'<div class="alert alert-error">{q_dict["err"][0]}</div>'

        html = HTML_HEADER + f"""
        <h1>Class Post <span style="font-size:0.6em; color:#666;">生徒用</span></h1>
        {msg}
        
        <div class="upload-area">
            <h3>📤 課題の提出</h3>
            <p style="color:#666;">作成したファイルをここに提出してください。</p>
            <form action="/submit_file" method="post" enctype="multipart/form-data">
                <label style="font-weight:bold; display:block; margin-bottom:5px;">氏名 (必須)</label>
                <input type="text" name="student_name" required placeholder="例：山田 太郎">
                <label style="font-weight:bold; display:block; margin-bottom:5px;">ファイル選択</label>
                <input type="file" name="file" required>
                <div style="margin-top:20px;">
                    <button type="submit" class="btn btn-primary" style="min-width:200px;">送信する</button>
                </div>
            </form>
        </div>

        <h3>📂 配布資料（DL）</h3>
        <ul class="card-list">
            {file_list_html if files else "<li style='padding:10px;'>配布資料はありません</li>"}
        </ul>
        """ + HTML_FOOTER
        self.send_response(200)
        self.send_header('Content-type', 'text/html; charset=utf-8')
        self.end_headers()
        self.wfile.write(html.encode('utf-8'))

    def render_teacher_page(self):
        config = load_config()
        if os.path.exists(SHARED_DIR):
            shared_files = sorted(os.listdir(SHARED_DIR))
        else:
            shared_files = []
        if os.path.exists(SUBMIT_DIR):
            submitted_files = sorted(os.listdir(SUBMIT_DIR))
        else:
            submitted_files = []

        shared_html = ""
        for f in shared_files:
            if f.startswith('.'): continue
            is_locked = f in config['locked_files']
            status = '<span class="status-tag tag-closed">非公開中</span>' if is_locked else '<span class="status-tag tag-open">公開中</span>'
            btn_text = "公開する" if is_locked else "非公開にする"
            btn_cls = "btn-success" if is_locked else "btn-secondary"
            bg_style = 'background:#fff3cd;' if is_locked else ''
            
            shared_html += f"""
            <li class="file-card" style="{bg_style}">
                <div class="file-info">
                    <span class="file-icon">📄</span>
                    <div>
                        <div>{f}</div>
                        {status}
                    </div>
                </div>
                <form action="/toggle_permission" method="post" style="display:inline;">
                    <input type="hidden" name="pwd" value="{ADMIN_PASSWORD}">
                    <input type="hidden" name="filename" value="{f}">
                    <button type="submit" class="btn {btn_cls}" style="font-size:13px; padding:8px 15px;">{btn_text}</button>
                </form>
            </li>
            """

        submitted_html = ""
        for f in submitted_files:
            if f.startswith('.'): continue
            dl_link = f'/submitted/{urllib.parse.quote(f)}?pwd={ADMIN_PASSWORD}'
            submitted_html += f'''
            <li class="file-card">
                <div class="file-info"><span class="file-icon">📝</span> {f}</div>
                <a href="{dl_link}" class="btn btn-primary" style="font-size:14px; width:45px; height:45px; display:inline-flex; align-items:center; justify-content:center; border-radius:50%; padding:0; flex-shrink:0;">DL</a>
            </li>'''

        html = HTML_HEADER + f"""
        <div class="nav-top">
            <span style="font-weight:bold; color:#e74c3c;">🔴 管理者モード</span>
            <form action="/shutdown" method="post" style="display:inline;" onsubmit="return confirm('サーバーを停止しますか？');">
                <input type="hidden" name="pwd" value="{ADMIN_PASSWORD}">
                <button type="submit" class="btn btn-danger" style="font-size:14px;">電源OFF</button>
            </form>
        </div>
        <h1>Class Post <span style="font-size:0.6em; color:#666;">教員用</span></h1>

        <div style="display: grid; grid-template-columns: 1fr 1fr; gap: 40px;">
            <div>
                <h3 style="background:#eef6fc; padding:10px; border-radius:5px;">📂 配布資料の管理</h3>
                <div class="upload-area" style="padding:15px; margin-bottom:15px;">
                    <p style="margin:0 0 10px 0; font-size:14px;">Webからファイルを追加</p>
                    <form action="/upload_material" method="post" enctype="multipart/form-data">
                        <input type="hidden" name="pwd" value="{ADMIN_PASSWORD}">
                        <input type="file" name="file" required style="margin-bottom:10px;">
                        <button type="submit" class="btn btn-primary" style="width:100%;">アップロード</button>
                    </form>
                </div>
                <ul class="card-list">
                    {shared_html if shared_files else "<li>ファイルがありません</li>"}
                </ul>
            </div>
            
            <div style="border-left: 2px solid #eee; padding-left: 20px;">
                <h3 style="background:#e8f5e9; padding:10px; border-radius:5px;">📥 生徒からの提出物</h3>
                <p>保存場所: <code>提出ボックス</code></p>
                <ul class="card-list">
                    {submitted_html if submitted_files else "<li>提出物はまだありません</li>"}
                </ul>
            </div>
        </div>
        
        <div style="margin-top:40px; text-align:center;">
             <a href="/" target="_blank" class="btn btn-secondary">別タブで生徒用画面を確認する</a>
        </div>
        """ + HTML_FOOTER
        self.send_response(200)
        self.send_header('Content-type', 'text/html; charset=utf-8')
        self.end_headers()
        self.wfile.write(html.encode('utf-8'))

class ThreadingHTTPServer(socketserver.ThreadingMixIn, http.server.HTTPServer):
    """マルチスレッド対応HTTPサーバー"""
    daemon_threads = True

    def handle_error(self, request, client_address):
        """リクエスト処理中のエラーをログファイルに記録（コンソールに出さない）"""
        try:
            traceback.print_exc()
        except Exception:
            pass


def update_student_shortcut(ip, port):
    """生徒用のショートカットHTML（生徒用.html）を自動更新する"""
    html_content = f"""<!DOCTYPE html>
<html lang="ja">
<head>
    <meta charset="UTF-8">
    <title>Class Post 接続</title>
    <style>
        body {{ font-family: 'Segoe UI', 'Meiryo', sans-serif; background: #f0f2f5; display: flex; align-items: center; justify-content: center; height: 100vh; margin: 0; }}
        .card {{ background: white; padding: 40px; border-radius: 20px; text-align: center; max-width: 420px; width: 90%; box-shadow: 0 10px 30px rgba(0,0,0,0.1); }}
        h1 {{ color: #2c3e50; font-size: 24px; margin: 0 0 8px 0; }}
        .subtitle {{ color: #7f8c8d; font-size: 14px; margin: 0 0 28px 0; }}
        .input-row {{ display: flex; gap: 8px; margin-bottom: 20px; }}
        .input-group {{ text-align: left; }}
        .input-group label {{ display: block; font-size: 11px; font-weight: 700; color: #7f8c8d; margin-bottom: 4px; text-transform: uppercase; letter-spacing: 0.05em; }}
        .input-group input {{ padding: 12px; font-size: 16px; border: 2px solid #e0e0e0; border-radius: 10px; font-family: monospace; outline: none; width: 100%; box-sizing: border-box; }}
        .input-group input:focus {{ border-color: #3498db; box-shadow: 0 0 0 3px rgba(52,152,219,0.15); }}
        .input-ip {{ flex: 3; }}
        .input-port {{ flex: 1; min-width: 80px; }}
        .url-preview {{ background: #f8f9fa; padding: 10px; border-radius: 8px; font-family: monospace; font-size: 14px; color: #3498db; margin-bottom: 20px; word-break: break-all; }}
        .btn {{ display: block; width: 100%; padding: 14px; background: #3498db; color: white; border: none; border-radius: 50px; font-size: 18px; font-weight: 600; cursor: pointer; transition: all 0.2s; box-shadow: 0 4px 12px rgba(52,152,219,0.3); }}
        .btn:hover {{ background: #2980b9; transform: translateY(-1px); box-shadow: 0 6px 16px rgba(52,152,219,0.4); }}
        .footer {{ margin-top: 20px; font-size: 12px; color: #e74c3c; background: #fdf2f2; padding: 10px; border-radius: 8px; }}
    </style>
</head>
<body>
    <div class="card">
        <h1>Class Post</h1>
        <p class="subtitle">接続先を確認してボタンを押してください</p>

        <div class="input-row">
            <div class="input-group input-ip">
                <label>IPアドレス</label>
                <input type="text" id="ip" value="{ip}" spellcheck="false">
            </div>
            <div class="input-group input-port">
                <label>ポート</label>
                <input type="text" id="port" value="{port}">
            </div>
        </div>

        <div class="url-preview" id="preview">http://{ip}:{port}/</div>

        <button class="btn" onclick="connect()">接続する</button>

        <div class="footer">※ 先生と同じWi-Fiに接続してから押してください</div>
    </div>
    <div style="position:fixed; bottom:4px; right:8px; font-size:9px; color:#ededee;">​seyaseya.org</div>
    <script>
        var ipEl = document.getElementById('ip');
        var portEl = document.getElementById('port');
        var previewEl = document.getElementById('preview');
        function updatePreview() {{
            previewEl.textContent = 'http://' + ipEl.value.trim() + ':' + portEl.value.trim() + '/';
        }}
        ipEl.addEventListener('input', updatePreview);
        portEl.addEventListener('input', updatePreview);
        function connect() {{
            var ip = ipEl.value.trim();
            var port = portEl.value.trim();
            if (!ip) {{ alert('IPアドレスを入力してください'); return; }}
            if (!port) {{ alert('ポート番号を入力してください'); return; }}
            window.location.href = 'http://' + ip + ':' + port + '/';
        }}
        ipEl.addEventListener('keydown', function(e) {{ if (e.key === 'Enter') connect(); }});
        portEl.addEventListener('keydown', function(e) {{ if (e.key === 'Enter') connect(); }});
    </script>
</body>
</html>
"""
    try:
        file_path = os.path.join(ROOT_DIR, '生徒用.html')
        with open(file_path, 'w', encoding='utf-8') as f:
            f.write(html_content)
    except Exception:
        pass

def run():
    global CURRENT_PORT
    ip = get_ip_address()
    hostname = socket.gethostname()
    port = DEFAULT_PORT
    httpd = None

    # ポート自動インクリメント
    while port < DEFAULT_PORT + 100:
        try:
            httpd = ThreadingHTTPServer(('0.0.0.0', port), ClassroomHandler)
            CURRENT_PORT = port
            break
        except OSError:
            port += 1

    if httpd:
        # 生徒用の接続用HTMLを自動更新
        update_student_shortcut(ip, CURRENT_PORT)

        safe_print("=" * 60)
        safe_print("  Class Post - Started!")
        safe_print("")
        safe_print(f"  Port: {CURRENT_PORT}")
        safe_print(f"  URL:  http://{ip}:{CURRENT_PORT}/")
        safe_print("")
        safe_print("  (Close this window to stop)")
        safe_print("=" * 60)

        # サーバー起動後にブラウザで管理画面を自動で開く
        def open_browser():
            try:
                webbrowser.open(f'http://localhost:{CURRENT_PORT}/manager?pwd={ADMIN_PASSWORD}')
            except Exception:
                pass
        threading.Timer(1.5, open_browser).start()

        try:
            httpd.serve_forever()
        except KeyboardInterrupt:
            pass
        except Exception:
            try:
                traceback.print_exc()
            except Exception:
                pass
        finally:
            httpd.server_close()
    else:
        safe_print("No available ports. Cannot start server.")


if __name__ == '__main__':
    run()